﻿//
//	app module
//
//	This is a top-level module for loading and initializing rat and the app.
//

//	this license block appears with other comments stripped in compiled version.
/**
* @license
* template, Copyright whoever.  All Rights Reserved.
* Rat is owned by Wahoo Studios, Inc.  See rat.js license.
* Firebase (sometimes embedded below in compiled version) is owned by Firebase, Inc.  See license URL below.
*/

//	global app object for this project
var app = {
	release : false,	//	use this to globally flag things like turning off cheats.
	
	types : {},			//	a list of types/constructors, so they're not all in global space
};

//	You don't have to use rat's module system if you don't want.
//	You can just include your code and go, as long as you somewhere call rat.load and rat.init.
//	But this is a good example of how to load modules through rat's module system with dependency tracking.
//
rat.modules.add( "js.app",
[	//	depends on these modules:
	
	//	rat
	"rat.debug.r_console",
	"rat.os.r_system",
	"rat.storage.r_storage",
	"rat.graphics.r_graphics",
	"rat.utils.r_utils",
	
	//	game
	"js.audio",
	"js.game",
	"js.actor",
	"js.data.data",
	
	"js.graphics.gfx",
	"js.graphics.effects",
	
	"js.ui.ui",
	"js.ui.main_menu",
	"js.ui.game_screen",
	"js.ui.ui_audio",
], 
function(rat)
{
	//	user settings/preferences stored locally
	app.settings = {
		version: 1,	//	current settings version
		soundOn : true,
		musicOn : true,
		
		admin : false,	//	useful flag to indicate (and store) that the player is a developer and has special access
		
		debug : {
			
		},
		
		//	add your own settings here
	};
	
	//	app config - display constants, debugging settings, etc.
	app.config = {
		
	};
	app.canvasElement = null;
	app.ctx = null;	//	global app rendering context - for convenient access
	app.state = 'loading';
	app.visibility = 'visible';
	
	//
	//	My one-time app init function.  Set up rat and some local things
	//
	app.init = function () {

		//console.log("--- app.init");
	
	    var canvasElement = document.getElementById("canvas");
	    app.canvasElement = canvasElement;
	
		app.ctx = canvasElement.getContext("2d");

		//	set up rat
		rat.init();
	
		//	start audio loading
		audio.init();
		
		//	init ui module
		ui.init();
		
		//	set up graphics behavior, scaling, etc.
		app.setupGraphics();
	
		//	rat will call our "draw" function below, each frame, before drawing UI
		rat.setDraw(app.draw);
		
		//	rat will call our "postDraw" function below, each frame, after drawing UI
		//rat.setPostUIDraw(app.postDraw);
		
		//	rat will call our "update" function below, each frame
		rat.setUpdate(app.update);

		//	handle window resizing
		rat.addEventListener('resize', app.resizeHandler);

		//	optional debugging help
		app.setupDebugging();
		
		//	Here is where you'd start preloading a bunch of images,
		//	which will mean there's time between startup and loading being complete.
		//	other modules (like gfx modules) may also add things to the preload list.
		rat.graphics.preLoadImages([
			//"blah.png",
		]);
		
		//	gfx system init
		gfx.oneTimeInit();
		//	effects system init
		effects.oneTimeInit();
		
		game.oneTimeInit();
		
		//	canvas starts empty, so background draws, which is some gray color,
		//	if we didn't change CSS.
		//	let's fill canvas to something more interesting for now.
		//	Even black is more interesting.
		app.ctx.fillStyle = "#000000";
		app.ctx.fillRect(0, 0, app.canvasElement.width, app.canvasElement.height);
		
		app.initTelemetry();
		
		app.storage = rat.storage.getStorage(rat.storage.permanentLocal);
		//	TEMPLATE: replate this prefix to match your own project's name
		app.storage.setPrefix("template_");
		app.readSettings();
		
		app.state = 'loading';
		
		//	track visibility for things like pausing music when tabbed out or mobile app closed
		rat.addEventListener('visibility', function(state) {
			app.visibility = state;
			audio.visibilityChange(state);
		});
		
		//console.log("--- app.init done");
		
		//	see initAfterLoad() for continuation...
	};
	
	//	After our initial loading is done, this is called, and we can initialize more stuff.
	app.initAfterLoad = function()
	{
		gfx.initAfterLoad();
		
		var start = 'main';	//	by default, go to main menu, or game, or whatever
		//	let settings override that, if somebody wants to store that value instead of changing code.
		if (app.settings.debug.skipTo)
			start = app.settings.debug.skipTo;
		
		app.gotoScreen(start);
	};
	
	//	init stats tracking (telemetry)
	app.initTelemetry = function()
	{
		//	TODO: re-add telemetry example with latest rat.telemetry functionality
	};
	
	//
	//	Set up console and other debugging aids
	//
	app.setupDebugging = function()
	{
		//	the rat console is pretty useful...
		if (!app.release)
			rat.console.allow(true);
		rat.console.setTextSize(32);
		
		//	here's how to start with the rat console up right away,
		//	which is helpful for debugging startup issues...
		//rat.console.activate(true);
		
		//	helpful debug display
		//	You can turn these on directly here,
		//	or just use the rat console commands like showfps, showstats
		//rat.system.debugDrawTiming = true;
		//rat.system.debugDrawStats = true;
		//rat.system.debugDrawFramerateGraph = true;
		
		//	register custom console cheats
		
		//	clear my settings
		rat.console.registerCommand("clearsettings", function (cmd, args)
		{
			app.clearSettings();
		}, ["reset", "resetsettings"]);
		
		//	set a "skipto" debug setting for starting up in various game states directly
		//	So, here's how this works...
		//	bring up the rat console ( / key)
		//	type something like "skipto game" to skip straight to the game on future reloads.
		//	reload to try it out.
		//	(it doesn't happen right away - it's a setting for what happens on future reloads)
		//	"skipto game" : skip to game
		//	"skipto main" : skip to main menu (or use any other screen code here)
		//	"skipto" : clear skipto value, e.g. to go back to normal behavior
		rat.console.registerCommand("skipTo", function (cmd, args)
		{
			var dest = args[0];
			if (!dest)
			{
				//	clear that debug setting entirely
				delete app.settings.debug.skipTo;
				app.writeSettings();
				return;
			}
			//	allow bounding quotes or not
			if (dest.charAt(0) === "'")
				dest = dest.substring(1, dest.length-1);
			app.skipTo(dest);
		}, ["skip"]);
		
	};
	
	//
	//	Set up display resolution and resize handling stuff
	//
	app.setupGraphics = function()
	{
		//	we have an "ideal target" resolution that all our screen math is based on
		//	(you can make this whatever you want - this happens to be a nice ipad-like resolution)
		var idealW = 2048;
		var idealH = 1536;
		
		//	additional scaling options
		var scaleOptions = {
			
			//	autoResizeCanvas lets rat resize to match display window, e.g. as the player resize their
			//	browser window, the game display is scaled up and down to match, but game logic can still
			//	use ideal values above, since scaling happens automatically in rat.
			autoResizeCanvas : true, //  in the web, yes, scale to ideal target
			
			//	allowAutoUpscale lets rat scale *bigger* than ideal target, e.g. on very large windows
			//	or high-res devices.
			allowAutoUpscale : true,	//	if window is bigger than ideal target
			
			//	Advanced: 
			//	autoFillOnResize is tricky:  We get as close as we can to the "ideal" target above,
			//	and when we resize our display space to match the aspect ratio of the window.
			//	at that point, rat.graphics.SCREEN_WIDTH and SCREEN_HEIGHT won't be our ideal values,
			//	and we'll have to recognize that (including any time it changes in resizeHandler())
			//	and adapt in code, e.g. draw health bars in bottom right corner correctly
			//	based on SCREEN_WIDTH and SCREEN_HEIGHT
			//	If you leave this false, then there'll be empty space to the right or bottom
			//	of the game's display area.
			autoFillOnResize : false,
		};
		rat.graphics.setAutoScaleFromIdeal(idealW, idealH, scaleOptions);
		
		//	autoclear:	For good performance, we should generally not be autoclearing,
		//			but this is really convenient for starting out a new project.
		//			TODO: decide on your own when to autoclear.
		rat.graphics.autoClearCanvas = true;
		rat.graphics.autoClearColor = "#0A4040";
		
		//	if we're doing any resize logic, run that once now
		app.resizeHandler();
		
	};
	
	//	go to a particular game screen or mode
	app.gotoScreen = function(screenID, args)
	{
		rat.screenManager.popAllScreens();

		var screen;
		
		if (screenID === 'main')
		{
			screen = new app.types.MainMenu();
		}
		/*else if (screenID === 'credits')
		{
			screen = new app.types.CreditsScreen();
		}
		*/
		//	etc.
		
		//	otherwise, jump straight to game
		//	(maybe bad screenID was specified in skipto debug param or something,
		//	so let's nicely default to game)
		if (screenID === 'game' || !screen)
		{
			app.gotoGame();
			return;
		}
		
		app.state = 'menus';
		if (screen)
			rat.screenManager.pushScreen(screen);
	};
	
	//	go to game mode.
	//	
	app.gotoGame = function()
	{
		game.init();
		app.state = 'play';
	};
	
	
	//	resize handler - called after rat does its thing.
	app.resizeHandler = function()
	{
		//console.log("resize handler");
		
		//	adjust/center UI, if we need to
		ui.resizeHandler();
	};
	
	//	repeating update function, if needed
	app.update = function(dt) {
	
		if (app.state === 'loading')
		{
			//	detect the completion of loading
			if (rat.graphics.isCacheLoaded() && rat.audio.isCacheLoaded())
			{
				//rat.console.log("Preload complete.");
				app.initAfterLoad();
			}
		}
		else if (app.state === 'play')
			game.update(dt);
		
		effects.update(dt);
		
		audio.update(dt);
		
		//app.stateTime += dt;
	};
	
	//	My draw function, mostly to handle loading state before everything's set up
	app.draw = function() {
		var ctx = app.ctx;
		
		//	use actual canvas size during loading - ignore rat setup?
		//var screenWidth = app.canvasElement.width;
		//var screenHeight = app.canvasElement.height;
		var screenWidth = rat.graphics.SCREEN_WIDTH;
		var screenHeight = rat.graphics.SCREEN_HEIGHT;
		
		if (app.state === 'loading')
		{
			//	custom loading rendering
			ctx.font = "76px " + ui.menus.mainFont.font;
			ctx.textAlign = "left";
			ctx.fillStyle = "#FFFFFF";
			ctx.fillText("Loading...", screenWidth * 0.3, screenHeight/2, 800);
		}
		else if (app.state === 'play')
			game.draw(ctx);
		
		//	feel free to move this
		effects.particles.draw();
	};
	
	//	draw some info about what version we think we are.
	//	This checks "versionInfo", which is empty by default,
	//	but can get set through our fancy compiled-game tool chain for real builds.
	app.drawVersion = function()
	{
		var verText = "v. ";
		if (app.versionInfo)
			verText += app.versionInfo.version + " " + app.versionInfo.date;
		else
			verText += " ?";
		var ctx = rat.graphics.getContext();
		ctx.fillStyle = "rgba(250, 250, 250, 0.5)";
		var x = 20;
		var y = rat.graphics.SCREEN_HEIGHT - 25;
		
		if (rat.system.kong)
		{
			if (rat.system.kong.fake)
				verText += " F";
		}
		
		ctx.font = "24px " + ui.mainFontName;
		ctx.textAlign = "left";
		ctx.fillText(verText, x, y, 800);
	};
	
	//	example of drawing on top of all UI elements, if needed
	/*
	app.postDraw = function()
	{
		if (app.state === 'loading')
		{
			//	draw on top of ui during loading? not really needed.
		}
		else if (app.state === 'play')
		{
			//	not really needed - game has its own postdraw setup
		} else {
			//effects.uiParticles.draw();
			//	optionally, draw version info on top of all menus?  might want to do this in main menu only
			//app.drawVersion();
		}
	};
	*/
	
	//	debug tool to change skipto and write it out to prefs.
	app.skipTo = function(what)
	{
		//if (!app.settings.admin)
		//	return;
		app.settings.debug.skipTo = what;
		app.writeSettings();
		console.log("set skipTo to " + what);
	};
	
	//	write settings
	app.writeSettings = function()
	{
		app.storage.setObject('settings', app.settings);
	};
	
	//	read settings
	app.readSettings = function()
	{
		rat.console.log("read settings");
		var settings = app.storage.getObject('settings');
		var updateSettings = false;	//	do we need to write back out after a fix?
		if (settings) {
			app.settings = settings;
		} else {
			rat.console.log("no settings");
			updateSettings = true;
		}
		
		//	make up a user ID that we can use to uniquely identify us for various things.
		if (typeof(app.settings.userID) === 'undefined')
		{
			var d = new Date();
			var m = d.getUTCMonth();
			var day = d.getUTCDate();
			var prefix = '' + (d.getUTCFullYear() - 2000) + ((m < 9) ? '0' : '') + (m+1) + ((day < 10) ? '0' : '') + day;
			
			app.settings.userID = prefix + '_' + rat.utils.makeID(10);
			rat.console.log("new user id " + app.settings.userID);
			
			updateSettings = true;
		}
		
		//	another example of how to fill in default value for missing var
		/*
		if (typeof(app.settings.scores) === 'undefined')
		{
			app.settings.scores = [];
			updateSettings = true;
		}
		*/
		if (typeof(app.settings.debug) === 'undefined')
		{
			app.settings.debug = {};
			updateSettings = true;
		}
		
		//	update right now so we don't have these missing fields in the future
		if (updateSettings)
			app.writeSettings();

		//	update system sound setting based on settings
		ui.updateAudioButtons();
		rat.audio.soundOn = app.settings.soundOn;
	};
	
});	//	end of rat.modules.add